#if NUNIT

#region References

using System;
using System.Collections;
using System.Data;
using System.Text;

using NUnit.Framework;

using gov.va.med.vbecs.DAL.HL7.OpenLibrary;
using gov.va.med.vbecs.DAL.HL7.OpenLibrary.Messages;
using gov.va.med.vbecs.DAL.HL7.Parsers;
using gov.va.med.vbecs.DAL.HL7AL;
using gov.va.med.vbecs.Common;

using ARTIFICIAL = gov.va.med.vbecs.Common.DatabaseConstants.ArtificialColumnNames;
using TABLE = gov.va.med.vbecs.Common.VbecsTables;
using SPROC = gov.va.med.vbecs.Common.VbecsStoredProcs;

using gov.va.med.vbecs.UnitTests;

#endregion

namespace gov.va.med.vbecs.DAL.HL7.UnitTests.Parsers
{
	[TestFixture]
	public class Parsers_PatientUpdateHL7Parser: BaseTest 
	{
		#region Variables

		private const string CARRIAGE_RETURN = "\x0D";
		//
		private const string MSH_A01 = "MSH^~|\\&^VAFC PIMS^589^VBECS ADT^589^20110407120005-0500^^ADT~A01^5894298121^T^2.3^^^NE^NE^USA";
		private const string MSH_A02 = "MSH^~|\\&^VAFC PIMS^589^VBECS ADT^589^20110407120005-0500^^ADT~A02^5894298121^T^2.3^^^NE^NE^USA";
		private const string MSH_A03 = "MSH^~|\\&^VAFC PIMS^589^VBECS ADT^589^20110407120005-0500^^ADT~A03^5894298121^T^2.3^^^NE^NE^USA";
		private const string MSH_A04 = "MSH^~|\\&^VAFC PIMS^589^VBECS ADT^589^20110407120005-0500^^ADT~A04^5894298121^T^2.3^^^NE^NE^USA";
		private const string MSH_A08 = "MSH^~|\\&^VAFC PIMS^589^VBECS ADT^589^20110407120005-0500^^ADT~A08^5894298121^T^2.3^^^NE^NE^USA";
		private const string MSH_A11 = "MSH^~|\\&^VAFC PIMS^589^VBECS ADT^589^20110407120005-0500^^ADT~A11^5894298121^T^2.3^^^NE^NE^USA";
		private const string MSH_A12 = "MSH^~|\\&^VAFC PIMS^589^VBECS ADT^589^20110407120005-0500^^ADT~A12^5894298121^T^2.3^^^NE^NE^USA";
		private const string MSH_A13 = "MSH^~|\\&^VAFC PIMS^589^VBECS ADT^589^20110407120005-0500^^ADT~A13^5894298121^T^2.3^^^NE^NE^USA";
		//
		private const string EVN_A01 = "EVN^A01^20110301135651-0500^^05";
		private const string EVN_A02 = "EVN^A02^20110407^^^USER,ONE";
		private const string EVN_A03 = "EVN^A03^20110407^^^USER,ONE";
		private const string EVN_A04 = "EVN^A04^20110301135651-0500^^05";
		private const string EVN_A08 = "EVN^A08^20110407^^^USER,ONE";
		private const string EVN_A11 = "EVN^A11^20110407^^^USER,ONE";
		private const string EVN_A12 = "EVN^A12^20110301135651-0500^^05";
		private const string EVN_A13 = "EVN^A13^20110407^^^USER,ONE";
		//
		private const string PV1_I = "PV1^1^I^10W P-KC~\"\"~\"\"^^^10W P-KC~C1016~2^209~USER~TWENTYTWO~~~~CSW^^^15^^^^^^^^NSC VETERAN^^^70^^^^^^^^^^^^^^^^^^589^^^^^20080506120416-0500^\"\"^^^^^1307736";
		private const string PV1_O = "PV1^1^O^\"\"^^^^^^^^^^^^^^^NON-VETERAN (OTHER)^^^^^^^^^^^^^^^^^^^^^^^^^^20110404^^^^^^1307964";
		//
		private string _fromPatientDfn;
		private string _fromPatientDob;
		private string _fromPatientDod;
		private string _fromPatientFirstName;
		private string _fromPatientIcn;
		private string _fromPatientLastName;
		private string _fromPatientMiddleName;
		private string _fromPatientSexCode;
		private string _fromPatientSsn;
		//
//		private string _fromPatientPidSegment;
		private string _toPatientPidSegment;
		//
		private HL7Interface _hl7Interface;
		//
		private Guid _patientGuid;

		#endregion

		#region Setup & TearDown

		[SetUp]
		public void SetUp()
		{
		    UnitTestUtil.RunSQL("DELETE FROM PatientLocation");     //TODO: Fix this lazy approach, tests create duplicate patient locations
			_hl7Interface = new HL7Interface( InterfaceName.Patient_Update.ToString() );
			//
			Common.LogonUser.LogonUserDUZ = "53311";
			Common.LogonUser.LogonUserDivisionCode = UnitTestConfig.TestDivisionCode;
		}

		#endregion

		/// <summary>
		/// ParseHL7Message - PASS (A01)
		/// </summary>
		[Test]
		public void ParseHL7Message_PASS_ADT_A01()
		{
			StringBuilder adtBuilder = new StringBuilder();
			//
			adtBuilder.Append( MSH_A01 );
			adtBuilder.Append( CARRIAGE_RETURN );
			adtBuilder.Append( EVN_A01 );
			adtBuilder.Append( CARRIAGE_RETURN );
			adtBuilder.Append( GetLocationUpdatePidSegment(true) );
			adtBuilder.Append( CARRIAGE_RETURN );
			adtBuilder.Append( PV1_I);
			adtBuilder.Append( CARRIAGE_RETURN );
			//
			HL7ProtocolMessage message = PatientUpdateHL7Parser.ParseHL7Message( _hl7Interface, adtBuilder.ToString() );
            Assert.IsNotNull(message, "PatientUpdate Message Parsed");
			Assert.AreEqual("AA", HL7Utility.ParseGetAckCode(message.GetMessage()) , "Ack Check" );
		}

		/// <summary>
		/// ParseHL7Message - PASS (A02)
		/// </summary>
		[Test]
		public void ParseHL7Message_PASS_ADT_A02()
		{
			StringBuilder adtBuilder = new StringBuilder();
			//
			adtBuilder.Append( MSH_A02 );
			adtBuilder.Append( CARRIAGE_RETURN );
			adtBuilder.Append( EVN_A02 );
			adtBuilder.Append( CARRIAGE_RETURN );
			adtBuilder.Append( GetLocationUpdatePidSegment(false) );
			adtBuilder.Append( CARRIAGE_RETURN );
			adtBuilder.Append( PV1_I);
			adtBuilder.Append( CARRIAGE_RETURN );
			//
			HL7ProtocolMessage message = PatientUpdateHL7Parser.ParseHL7Message( _hl7Interface, adtBuilder.ToString() );
			Assert.IsNotNull( message, "PatientUpdate Message Parsed" );
			Assert.AreEqual("AA", HL7Utility.ParseGetAckCode(message.GetMessage()) , "Ack Check" );
		}

		/// <summary>
		/// ParseHL7Message - PASS (A03)
		/// </summary>
		[Test]
		public void ParseHL7Message_PASS_ADT_A03()
		{
			StringBuilder adtBuilder = new StringBuilder();
			//
			adtBuilder.Append( MSH_A03 );
			adtBuilder.Append( CARRIAGE_RETURN );
			adtBuilder.Append( EVN_A03 );
			adtBuilder.Append( CARRIAGE_RETURN );
			adtBuilder.Append( GetLocationUpdatePidSegment(true) );
			adtBuilder.Append( CARRIAGE_RETURN );
			adtBuilder.Append( PV1_I);
			adtBuilder.Append( CARRIAGE_RETURN );
			//
			HL7ProtocolMessage message = PatientUpdateHL7Parser.ParseHL7Message( _hl7Interface, adtBuilder.ToString() );
            Assert.IsNotNull(message, "PatientUpdate Message Parsed");
			Assert.AreEqual("AA", HL7Utility.ParseGetAckCode(message.GetMessage()) , "Ack CHeck" );
		}

		/// <summary>
		/// ParseHL7Message - PASS (A04)
		/// </summary>
		[Test]
		public void ParseHL7Message_PASS_ADT_A04()
		{
			// *************************************************************************************************************************
			// TEST #1: CHANGE FIRST NAME
			GetPatientData();
			//
			string toPatientDob = _fromPatientDob;
			string toPatientDod = _fromPatientDod;
			//
			string toPatientFirstName = "AAAAAA";
			if ( _fromPatientFirstName != null && _fromPatientFirstName != string.Empty && _fromPatientFirstName.Length > 0 )
			{
				// ex. LISA -> ALIS
				toPatientFirstName = string.Concat( _fromPatientFirstName[_fromPatientFirstName.Length - 1], _fromPatientFirstName.Substring( 1, _fromPatientFirstName.Length - 1) );
			}
			//
			string toPatientLastName = _fromPatientLastName;
			string toPatientMiddleName = _fromPatientMiddleName;
			// Sex Code is required in the message but our DB may have null values so let's make sure we are safe
			string toPatientSexCode = _fromPatientSexCode == null ? "F" : _fromPatientSexCode;
			string toPatientSsn = _fromPatientSsn;
			//
			_toPatientPidSegment = BuildPid( _fromPatientIcn, _fromPatientDfn, toPatientLastName, toPatientFirstName, toPatientMiddleName, toPatientDob, toPatientSexCode, toPatientSsn, toPatientDod );
			//
			StringBuilder adtBuilder = new StringBuilder();
			//
			adtBuilder.Append( MSH_A04 );
			adtBuilder.Append( CARRIAGE_RETURN );
			adtBuilder.Append( EVN_A04 );
			adtBuilder.Append( CARRIAGE_RETURN );
			adtBuilder.Append( _toPatientPidSegment );
			adtBuilder.Append( CARRIAGE_RETURN );
			adtBuilder.Append( PV1_I);
			adtBuilder.Append( CARRIAGE_RETURN );
			//
			HL7ProtocolMessage message = PatientUpdateHL7Parser.ParseHL7Message( _hl7Interface, adtBuilder.ToString() );
            Assert.IsNotNull(message, "PatientUpdate Message Parsed");
			Assert.AreEqual("AA", HL7Utility.ParseGetAckCode(message.GetMessage()) , "Ack Check" );
			//
			// Ensure data was persisted correctly
			GetPatientData();
			Assert.AreEqual( _fromPatientFirstName , toPatientFirstName, "First Name" );
			//
			// *************************************************************************************************************************
			// *************************************************************************************************************************
			// *************************************************************************************************************************
			// TEST #2: CHANGE MIDDLE NAME
			//
			adtBuilder.Remove( 0, adtBuilder.Length );
			//
			toPatientDob = _fromPatientDob;
			toPatientDod = _fromPatientDod;
			toPatientFirstName = _fromPatientFirstName;
			toPatientLastName = _fromPatientLastName;
			//
			if ( _fromPatientMiddleName != null && _fromPatientMiddleName != string.Empty && _fromPatientMiddleName.Length > 0 )
			{
				toPatientMiddleName = string.Concat( _fromPatientMiddleName[_fromPatientMiddleName.Length - 1], _fromPatientMiddleName.Substring( 1, _fromPatientMiddleName.Length - 1) );
			}
			else
			{
				toPatientMiddleName = "BBBBBBB";
			}
			//
			// Sex Code is required in the message but our DB may have null values so let's make sure we are safe
			toPatientSexCode = _fromPatientSexCode == null ? "F" : _fromPatientSexCode;
			toPatientSsn = _fromPatientSsn;
			//
			_toPatientPidSegment = BuildPid( _fromPatientIcn, _fromPatientDfn, toPatientLastName, toPatientFirstName, toPatientMiddleName, toPatientDob, toPatientSexCode, toPatientSsn, toPatientDod );
			//
			adtBuilder.Append( MSH_A04 );
			adtBuilder.Append( CARRIAGE_RETURN );
			adtBuilder.Append( EVN_A04 );
			adtBuilder.Append( CARRIAGE_RETURN );
			adtBuilder.Append( _toPatientPidSegment );
			adtBuilder.Append( CARRIAGE_RETURN );
			adtBuilder.Append( PV1_I);
			adtBuilder.Append( CARRIAGE_RETURN );
			//
			message = PatientUpdateHL7Parser.ParseHL7Message( _hl7Interface, adtBuilder.ToString() );
            Assert.IsNotNull(message, "PatientUpdate Message Parsed");
			Assert.AreEqual("AA", HL7Utility.ParseGetAckCode(message.GetMessage()) , "Ack Check" );
			//
			// Ensure data was persisted correctly
			GetPatientData();
			Assert.AreEqual( _fromPatientMiddleName , toPatientMiddleName, "Middle Name" );
			//
			// *************************************************************************************************************************
			// *************************************************************************************************************************
			// *************************************************************************************************************************
			// TEST #3: CHANGE LAST NAME
			//
			adtBuilder.Remove( 0, adtBuilder.Length );
			//
			toPatientDob = _fromPatientDob;
			toPatientDod = _fromPatientDod;
			toPatientFirstName = _fromPatientFirstName;
			//
			if ( _fromPatientLastName != null && _fromPatientLastName != string.Empty && _fromPatientLastName.Length > 0 )
			{
				toPatientLastName = string.Concat( _fromPatientLastName[_fromPatientLastName.Length - 1], _fromPatientLastName.Substring( 1, _fromPatientLastName.Length - 1) );
			}
			else
			{
				toPatientLastName = "CCCCCCC";
			}
			//
			toPatientMiddleName = _fromPatientMiddleName;
			// Sex Code is required in the message but our DB may have null values so let's make sure we are safe
			toPatientSexCode = _fromPatientSexCode == null ? "F" : _fromPatientSexCode;
			toPatientSsn = _fromPatientSsn;
			//
			_toPatientPidSegment = BuildPid( _fromPatientIcn, _fromPatientDfn, toPatientLastName, toPatientFirstName, toPatientMiddleName, toPatientDob, toPatientSexCode, toPatientSsn, toPatientDod );
			//
			adtBuilder.Append( MSH_A04 );
			adtBuilder.Append( CARRIAGE_RETURN );
			adtBuilder.Append( EVN_A04 );
			adtBuilder.Append( CARRIAGE_RETURN );
			adtBuilder.Append( _toPatientPidSegment );
			adtBuilder.Append( CARRIAGE_RETURN );
			adtBuilder.Append( PV1_I);
			adtBuilder.Append( CARRIAGE_RETURN );
			//
			message = PatientUpdateHL7Parser.ParseHL7Message( _hl7Interface, adtBuilder.ToString() );
            Assert.IsNotNull(message, "PatientUpdate Message Parsed");
			Assert.AreEqual("AA", HL7Utility.ParseGetAckCode(message.GetMessage()) , "Ack Check" );
			//
			// Ensure data was persisted correctly
			GetPatientData();
			Assert.AreEqual( _fromPatientLastName , toPatientLastName, "LastName" );
			//
			// *************************************************************************************************************************
			// *************************************************************************************************************************
			// *************************************************************************************************************************
			// TEST #4: CHANGE SEX CODE
			//
			adtBuilder.Remove( 0, adtBuilder.Length );
			//
			toPatientDob = _fromPatientDob;
			toPatientDod = _fromPatientDod;
			toPatientFirstName = _fromPatientFirstName;
			toPatientLastName = _fromPatientLastName;
			toPatientMiddleName = _fromPatientMiddleName;
			toPatientSexCode = _fromPatientSexCode != null && _fromPatientSexCode == "M" ? "F" : "M";
			toPatientSsn = _fromPatientSsn;
			//
			_toPatientPidSegment = BuildPid( _fromPatientIcn, _fromPatientDfn, toPatientLastName, toPatientFirstName, toPatientMiddleName, toPatientDob, toPatientSexCode, toPatientSsn, toPatientDod );
			//
			adtBuilder.Append( MSH_A04 );
			adtBuilder.Append( CARRIAGE_RETURN );
			adtBuilder.Append( EVN_A04 );
			adtBuilder.Append( CARRIAGE_RETURN );
			adtBuilder.Append( _toPatientPidSegment );
			adtBuilder.Append( CARRIAGE_RETURN );
			adtBuilder.Append( PV1_I);
			adtBuilder.Append( CARRIAGE_RETURN );
			//
			message = PatientUpdateHL7Parser.ParseHL7Message( _hl7Interface, adtBuilder.ToString() );
            Assert.IsNotNull(message, "PatientUpdate Message Parsed");
			Assert.AreEqual("AA", HL7Utility.ParseGetAckCode(message.GetMessage()) , "Ack CHeck" );
			//
			// Ensure data was persisted correctly
			GetPatientData();
			Assert.AreEqual( _fromPatientSexCode , toPatientSexCode, "Sex Code" );
			//
			// *************************************************************************************************************************
			// *************************************************************************************************************************
			// *************************************************************************************************************************
			// TEST #5: CHANGE SSN (then change it back)
			//
			adtBuilder.Remove( 0, adtBuilder.Length );
			//
			string origSsn = _fromPatientSsn;
			//
			toPatientDob = _fromPatientDob;
			toPatientDod = _fromPatientDod;
			toPatientFirstName = _fromPatientFirstName;
			toPatientLastName = _fromPatientLastName;
			toPatientMiddleName = _fromPatientMiddleName;
			// Sex Code is required in the message but our DB may have null values so let's make sure we are safe
			toPatientSexCode = _fromPatientSexCode == null ? "F" : _fromPatientSexCode;
			//
			DateTime now = DateTime.UtcNow;
			toPatientSsn = "101010101"; 
			//
			if ( _fromPatientSsn != null && _fromPatientSsn != string.Empty && _fromPatientSsn.Length > 0 )
			{
				toPatientSsn = string.Concat( now.Year, now.Hour.ToString().PadLeft(2, '0'), now.Minute.ToString().PadLeft(2, '0'), now.Minute.ToString().Substring(0, 1));
			}
			//
			_toPatientPidSegment = BuildPid( _fromPatientIcn, _fromPatientDfn, toPatientLastName, toPatientFirstName, toPatientMiddleName, toPatientDob, toPatientSexCode, toPatientSsn, toPatientDod );
			//
			adtBuilder.Append( MSH_A04 );
			adtBuilder.Append( CARRIAGE_RETURN );
			adtBuilder.Append( EVN_A04 );
			adtBuilder.Append( CARRIAGE_RETURN );
			adtBuilder.Append( _toPatientPidSegment );
			adtBuilder.Append( CARRIAGE_RETURN );
			adtBuilder.Append( PV1_I);
			adtBuilder.Append( CARRIAGE_RETURN );
			//
			message = PatientUpdateHL7Parser.ParseHL7Message( _hl7Interface, adtBuilder.ToString() );
            Assert.IsNotNull(message, "PatientUpdate Message Parsed");
			Assert.AreEqual("AA", HL7Utility.ParseGetAckCode(message.GetMessage()) , "Ack Check" );
			//
			// Ensure data was persisted correctly
			GetPatientData();
			Assert.AreEqual( _fromPatientSsn , toPatientSsn, "SSN" );
			//
			// Set the SSN back to what it was ...
			//
			adtBuilder.Remove( 0, adtBuilder.Length );
			GetPatientData();
			//
			_toPatientPidSegment = BuildPid( _fromPatientIcn, _fromPatientDfn, _fromPatientLastName, _fromPatientFirstName, _fromPatientMiddleName, _fromPatientDob, _fromPatientSexCode, origSsn, _fromPatientDod );
			//
			adtBuilder.Append( MSH_A04 );
			adtBuilder.Append( CARRIAGE_RETURN );
			adtBuilder.Append( EVN_A04 );
			adtBuilder.Append( CARRIAGE_RETURN );
			adtBuilder.Append( _toPatientPidSegment );
			adtBuilder.Append( CARRIAGE_RETURN );
			adtBuilder.Append( PV1_I);
			adtBuilder.Append( CARRIAGE_RETURN );
			//
			message = PatientUpdateHL7Parser.ParseHL7Message( _hl7Interface, adtBuilder.ToString() );
            Assert.IsNotNull(message, "PatientUpdate Message Parsed");
			Assert.AreEqual("AA", HL7Utility.ParseGetAckCode(message.GetMessage()) , "Ack Check" );
			//
			GetPatientData();
			Assert.AreEqual( _fromPatientSsn , origSsn, "Orig SSN" );
			//
			// *************************************************************************************************************************
			// *************************************************************************************************************************
			// *************************************************************************************************************************
			// TEST #6: CHANGE DOB (then change it back)
			//
			adtBuilder.Remove( 0, adtBuilder.Length );
			//
			string origDob = _fromPatientDob;
			//
			toPatientFirstName = _fromPatientFirstName;
			toPatientLastName = _fromPatientLastName;
			toPatientMiddleName = _fromPatientMiddleName;
			// Sex Code is required in the message but our DB may have null values so let's make sure we are safe
			toPatientSexCode = _fromPatientSexCode == null ? "F" : _fromPatientSexCode;
			toPatientDod = _fromPatientDod;
			//
			toPatientDob = Common.HL7DateFormat.ConvertDateTimeToHL7(DateTime.UtcNow.Subtract(new TimeSpan(7777, 77, 77, 77, 77)));
			//
			_toPatientPidSegment = BuildPid( _fromPatientIcn, _fromPatientDfn, toPatientLastName, toPatientFirstName, toPatientMiddleName, toPatientDob, toPatientSexCode, toPatientSsn, toPatientDod );
			//
			adtBuilder.Append( MSH_A04 );
			adtBuilder.Append( CARRIAGE_RETURN );
			adtBuilder.Append( EVN_A04 );
			adtBuilder.Append( CARRIAGE_RETURN );
			adtBuilder.Append( _toPatientPidSegment );
			adtBuilder.Append( CARRIAGE_RETURN );
			adtBuilder.Append( PV1_I);
			adtBuilder.Append( CARRIAGE_RETURN );
			//
			message = PatientUpdateHL7Parser.ParseHL7Message( _hl7Interface, adtBuilder.ToString() );
            Assert.IsNotNull(message, "PatientUpdate Message Parsed");
			Assert.AreEqual("AA", HL7Utility.ParseGetAckCode(message.GetMessage()) , "Ack Check" );
			//
			// Ensure data was persisted correctly
			GetPatientData();
			Assert.AreEqual( _fromPatientDob , toPatientDob, "DOB" );
			//
			// Re-set Dob
			//
			adtBuilder.Remove( 0, adtBuilder.Length );
			GetPatientData();
			//
			_toPatientPidSegment = BuildPid( _fromPatientIcn, _fromPatientDfn, _fromPatientLastName, _fromPatientFirstName, _fromPatientMiddleName, origDob, _fromPatientSexCode, _fromPatientSsn, string.Empty );
			//
			adtBuilder.Append( MSH_A04 );
			adtBuilder.Append( CARRIAGE_RETURN );
			adtBuilder.Append( EVN_A04 );
			adtBuilder.Append( CARRIAGE_RETURN );
			adtBuilder.Append( _toPatientPidSegment );
			adtBuilder.Append( CARRIAGE_RETURN );
			adtBuilder.Append( PV1_I);
			adtBuilder.Append( CARRIAGE_RETURN );
			//
			message = PatientUpdateHL7Parser.ParseHL7Message( _hl7Interface, adtBuilder.ToString() );
            Assert.IsNotNull(message, "PatientUpdate Message Parsed");
			Assert.AreEqual("AA", HL7Utility.ParseGetAckCode(message.GetMessage()) , "Ack Check" );
			//
			GetPatientData();
			Assert.AreEqual( _fromPatientDob , origDob, "Orig DOB" );
			//
			// *************************************************************************************************************************
			// *************************************************************************************************************************
			// *************************************************************************************************************************
			// TEST #7: CHANGE DOD: ADD (then remove)
			//
			adtBuilder.Remove( 0, adtBuilder.Length );
			//
			toPatientFirstName = _fromPatientFirstName;
			toPatientLastName = _fromPatientLastName;
			toPatientMiddleName = _fromPatientMiddleName;
			// Sex Code is required in the message but our DB may have null values so let's make sure we are safe
			toPatientSexCode = _fromPatientSexCode == null ? "F" : _fromPatientSexCode;
			//
			if ( _fromPatientDod != string.Empty )
			{
				toPatientDod = string.Empty;
			}
			else
			{
				toPatientDod = Common.HL7DateFormat.ConvertDateTimeToHL7(DateTime.UtcNow);
			}
			//
			_toPatientPidSegment = BuildPid( _fromPatientIcn, _fromPatientDfn, toPatientLastName, toPatientFirstName, toPatientMiddleName, toPatientDob, toPatientSexCode, toPatientSsn, toPatientDod );
			//
			adtBuilder.Append( MSH_A04 );
			adtBuilder.Append( CARRIAGE_RETURN );
			adtBuilder.Append( EVN_A04 );
			adtBuilder.Append( CARRIAGE_RETURN );
			adtBuilder.Append( _toPatientPidSegment );
			adtBuilder.Append( CARRIAGE_RETURN );
			adtBuilder.Append( PV1_I);
			adtBuilder.Append( CARRIAGE_RETURN );
			//
			message = PatientUpdateHL7Parser.ParseHL7Message( _hl7Interface, adtBuilder.ToString() );
            Assert.IsNotNull(message, "PatientUpdate Message Parsed");
			Assert.AreEqual("AA", HL7Utility.ParseGetAckCode(message.GetMessage()) , "Ack Check" );
			//
			// Ensure data was persisted correctly
			GetPatientData();
			Assert.AreEqual( _fromPatientDod , toPatientDod, "DOD" );
			//
			// Remove Dod
			//
			adtBuilder.Remove( 0, adtBuilder.Length );
			GetPatientData();
			//
			_toPatientPidSegment = BuildPid( _fromPatientIcn, _fromPatientDfn, _fromPatientLastName, _fromPatientFirstName, _fromPatientMiddleName, _fromPatientDob, _fromPatientSexCode, _fromPatientSsn, string.Empty );
			//
			adtBuilder.Append( MSH_A04 );
			adtBuilder.Append( CARRIAGE_RETURN );
			adtBuilder.Append( EVN_A04 );
			adtBuilder.Append( CARRIAGE_RETURN );
			adtBuilder.Append( _toPatientPidSegment );
			adtBuilder.Append( CARRIAGE_RETURN );
			adtBuilder.Append( PV1_I);
			adtBuilder.Append( CARRIAGE_RETURN );
			//
			message = PatientUpdateHL7Parser.ParseHL7Message( _hl7Interface, adtBuilder.ToString() );
            Assert.IsNotNull(message, "PatientUpdate Message Parsed");
			Assert.AreEqual("AA", HL7Utility.ParseGetAckCode(message.GetMessage()) , "Ack Check" );
			//
			GetPatientData();
			Assert.AreEqual( _fromPatientDod , string.Empty, "Orig DOD" );
		}

		/// <summary>
		/// ParseHL7Message - PASS (A08)
		/// </summary>
		[Test]
		public void ParseHL7Message_PASS_ADT_A08()
		{
			// *************************************************************************************************************************
			// TEST #1: CHANGE FIRST NAME
			GetPatientData();
			//
			string toPatientDob = _fromPatientDob;
			string toPatientDod = _fromPatientDod;
			//
			string toPatientFirstName = "AAAAAA";
			if ( _fromPatientFirstName != null && _fromPatientFirstName != string.Empty && _fromPatientFirstName.Length > 0 )
			{
				// ex. LISA -> ALIS
				toPatientFirstName = string.Concat( _fromPatientFirstName[_fromPatientFirstName.Length - 1], _fromPatientFirstName.Substring( 1, _fromPatientFirstName.Length - 1) );
			}
			//
			string toPatientLastName = _fromPatientLastName;
			string toPatientMiddleName = _fromPatientMiddleName;
			// Sex Code is required in the message but our DB may have null values so let's make sure we are safe
			string toPatientSexCode = _fromPatientSexCode == null ? "F" : _fromPatientSexCode;
			string toPatientSsn = _fromPatientSsn;
			//
			_toPatientPidSegment = BuildPid( _fromPatientIcn, _fromPatientDfn, toPatientLastName, toPatientFirstName, toPatientMiddleName, toPatientDob, toPatientSexCode, toPatientSsn, toPatientDod );
			//
			StringBuilder adtBuilder = new StringBuilder();
			//
			adtBuilder.Append( MSH_A08 );
			adtBuilder.Append( CARRIAGE_RETURN );
			adtBuilder.Append( EVN_A08 );
			adtBuilder.Append( CARRIAGE_RETURN );
			adtBuilder.Append( _toPatientPidSegment );
			adtBuilder.Append( CARRIAGE_RETURN );
			adtBuilder.Append( PV1_O);
			adtBuilder.Append( CARRIAGE_RETURN );
			//
			HL7ProtocolMessage message = PatientUpdateHL7Parser.ParseHL7Message( _hl7Interface, adtBuilder.ToString() );
            Assert.IsNotNull(message, "PatientUpdate Message Parsed");
			Assert.AreEqual("AA", HL7Utility.ParseGetAckCode(message.GetMessage()) , "Ack Check" );
			//
			// Ensure data was persisted correctly
			GetPatientData();
			Assert.AreEqual( _fromPatientFirstName , toPatientFirstName, "First Name" );
			//
			// *************************************************************************************************************************
			// *************************************************************************************************************************
			// *************************************************************************************************************************
			// TEST #2: CHANGE MIDDLE NAME
			//
			adtBuilder.Remove( 0, adtBuilder.Length );
			//
			toPatientDob = _fromPatientDob;
			toPatientDod = _fromPatientDod;
			toPatientFirstName = _fromPatientFirstName;
			toPatientLastName = _fromPatientLastName;
			//
			if ( _fromPatientMiddleName != null && _fromPatientMiddleName != string.Empty && _fromPatientMiddleName.Length > 0 )
			{
				toPatientMiddleName = string.Concat( _fromPatientMiddleName[_fromPatientMiddleName.Length - 1], _fromPatientMiddleName.Substring( 1, _fromPatientMiddleName.Length - 1) );
			}
			else
			{
				toPatientMiddleName = "BBBBBBB";
			}
			//
			// Sex Code is required in the message but our DB may have null values so let's make sure we are safe
			toPatientSexCode = _fromPatientSexCode == null ? "F" : _fromPatientSexCode;
			toPatientSsn = _fromPatientSsn;
			//
			_toPatientPidSegment = BuildPid( _fromPatientIcn, _fromPatientDfn, toPatientLastName, toPatientFirstName, toPatientMiddleName, toPatientDob, toPatientSexCode, toPatientSsn, toPatientDod );
			//
			adtBuilder.Append( MSH_A08 );
			adtBuilder.Append( CARRIAGE_RETURN );
			adtBuilder.Append( EVN_A08 );
			adtBuilder.Append( CARRIAGE_RETURN );
			adtBuilder.Append( _toPatientPidSegment );
			adtBuilder.Append( CARRIAGE_RETURN );
			adtBuilder.Append( PV1_O);
			adtBuilder.Append( CARRIAGE_RETURN );
			//
			message = PatientUpdateHL7Parser.ParseHL7Message( _hl7Interface, adtBuilder.ToString() );
            Assert.IsNotNull(message, "PatientUpdate Message Parsed");
			Assert.AreEqual("AA", HL7Utility.ParseGetAckCode(message.GetMessage()) , "Ack Check" );
			//
			// Ensure data was persisted correctly
			GetPatientData();
			Assert.AreEqual( _fromPatientMiddleName , toPatientMiddleName, "Middle Name" );
			//
			// *************************************************************************************************************************
			// *************************************************************************************************************************
			// *************************************************************************************************************************
			// TEST #3: CHANGE LAST NAME
			//
			adtBuilder.Remove( 0, adtBuilder.Length );
			//
			toPatientDob = _fromPatientDob;
			toPatientDod = _fromPatientDod;
			toPatientFirstName = _fromPatientFirstName;
			//
			if ( _fromPatientLastName != null && _fromPatientLastName != string.Empty && _fromPatientLastName.Length > 0 )
			{
				toPatientLastName = string.Concat( _fromPatientLastName[_fromPatientLastName.Length - 1], _fromPatientLastName.Substring( 1, _fromPatientLastName.Length - 1) );
			}
			else
			{
				toPatientLastName = "CCCCCCC";
			}
			//
			toPatientMiddleName = _fromPatientMiddleName;
			// Sex Code is required in the message but our DB may have null values so let's make sure we are safe
			toPatientSexCode = _fromPatientSexCode == null ? "F" : _fromPatientSexCode;
			toPatientSsn = _fromPatientSsn;
			//
			_toPatientPidSegment = BuildPid( _fromPatientIcn, _fromPatientDfn, toPatientLastName, toPatientFirstName, toPatientMiddleName, toPatientDob, toPatientSexCode, toPatientSsn, toPatientDod );
			//
			adtBuilder.Append( MSH_A08 );
			adtBuilder.Append( CARRIAGE_RETURN );
			adtBuilder.Append( EVN_A08 );
			adtBuilder.Append( CARRIAGE_RETURN );
			adtBuilder.Append( _toPatientPidSegment );
			adtBuilder.Append( CARRIAGE_RETURN );
			adtBuilder.Append( PV1_O);
			adtBuilder.Append( CARRIAGE_RETURN );
			//
			message = PatientUpdateHL7Parser.ParseHL7Message( _hl7Interface, adtBuilder.ToString() );
            Assert.IsNotNull(message, "PatientUpdate Message Parsed");
			Assert.AreEqual("AA", HL7Utility.ParseGetAckCode(message.GetMessage()) , "Ack Check" );
			//
			// Ensure data was persisted correctly
			GetPatientData();
			Assert.AreEqual( _fromPatientLastName , toPatientLastName, "Last Name" );
			//
			// *************************************************************************************************************************
			// *************************************************************************************************************************
			// *************************************************************************************************************************
			// TEST #4: CHANGE SEX CODE
			//
			adtBuilder.Remove( 0, adtBuilder.Length );
			//
			toPatientDob = _fromPatientDob;
			toPatientDod = _fromPatientDod;
			toPatientFirstName = _fromPatientFirstName;
			toPatientLastName = _fromPatientLastName;
			toPatientMiddleName = _fromPatientMiddleName;
			toPatientSexCode = _fromPatientSexCode != null && _fromPatientSexCode == "M" ? "F" : "M";
			toPatientSsn = _fromPatientSsn;
			//
			_toPatientPidSegment = BuildPid( _fromPatientIcn, _fromPatientDfn, toPatientLastName, toPatientFirstName, toPatientMiddleName, toPatientDob, toPatientSexCode, toPatientSsn, toPatientDod );
			//
			adtBuilder.Append( MSH_A08 );
			adtBuilder.Append( CARRIAGE_RETURN );
			adtBuilder.Append( EVN_A08 );
			adtBuilder.Append( CARRIAGE_RETURN );
			adtBuilder.Append( _toPatientPidSegment );
			adtBuilder.Append( CARRIAGE_RETURN );
			adtBuilder.Append( PV1_O);
			adtBuilder.Append( CARRIAGE_RETURN );
			//
			message = PatientUpdateHL7Parser.ParseHL7Message( _hl7Interface, adtBuilder.ToString() );
            Assert.IsNotNull(message, "PatientUpdate Message Parsed");
			Assert.AreEqual("AA", HL7Utility.ParseGetAckCode(message.GetMessage()) , "Ack Check" );
			//
			// Ensure data was persisted correctly
			GetPatientData();
			Assert.AreEqual( _fromPatientSexCode , toPatientSexCode, "Sex Code" );
			//
			// *************************************************************************************************************************
			// *************************************************************************************************************************
			// *************************************************************************************************************************
			// TEST #5: CHANGE SSN (then change it back)
			//
			adtBuilder.Remove( 0, adtBuilder.Length );
			//
			string origSsn = _fromPatientSsn;
			//
			toPatientDob = _fromPatientDob;
			toPatientDod = _fromPatientDod;
			toPatientFirstName = _fromPatientFirstName;
			toPatientLastName = _fromPatientLastName;
			toPatientMiddleName = _fromPatientMiddleName;
			// Sex Code is required in the message but our DB may have null values so let's make sure we are safe
			toPatientSexCode = _fromPatientSexCode == null ? "F" : _fromPatientSexCode;
			//
			DateTime now = DateTime.UtcNow;
			toPatientSsn = "101010101"; 
			//
			if ( _fromPatientSsn != null && _fromPatientSsn != string.Empty && _fromPatientSsn.Length > 0 )
			{
				toPatientSsn = string.Concat( now.Year, now.Hour.ToString().PadLeft(2, '0'), now.Minute.ToString().PadLeft(2, '0'), now.Minute.ToString().Substring(0, 1));
			}
			//
			_toPatientPidSegment = BuildPid( _fromPatientIcn, _fromPatientDfn, toPatientLastName, toPatientFirstName, toPatientMiddleName, toPatientDob, toPatientSexCode, toPatientSsn, toPatientDod );
			//
			adtBuilder.Append( MSH_A08 );
			adtBuilder.Append( CARRIAGE_RETURN );
			adtBuilder.Append( EVN_A08 );
			adtBuilder.Append( CARRIAGE_RETURN );
			adtBuilder.Append( _toPatientPidSegment );
			adtBuilder.Append( CARRIAGE_RETURN );
			adtBuilder.Append( PV1_O);
			adtBuilder.Append( CARRIAGE_RETURN );
			//
			message = PatientUpdateHL7Parser.ParseHL7Message( _hl7Interface, adtBuilder.ToString() );
            Assert.IsNotNull(message, "PatientUpdate Message Parsed");
			Assert.AreEqual("AA", HL7Utility.ParseGetAckCode(message.GetMessage()) , "Ack Check" );
			//
			// Ensure data was persisted correctly
			GetPatientData();
			Assert.AreEqual( _fromPatientSsn , toPatientSsn, "SSN" );
			//
			// Set the SSN back to what it was ...
			//
			adtBuilder.Remove( 0, adtBuilder.Length );
			GetPatientData();
			//
			_toPatientPidSegment = BuildPid( _fromPatientIcn, _fromPatientDfn, _fromPatientLastName, _fromPatientFirstName, _fromPatientMiddleName, _fromPatientDob, _fromPatientSexCode, origSsn, _fromPatientDod );
			//
			adtBuilder.Append( MSH_A08 );
			adtBuilder.Append( CARRIAGE_RETURN );
			adtBuilder.Append( EVN_A08 );
			adtBuilder.Append( CARRIAGE_RETURN );
			adtBuilder.Append( _toPatientPidSegment );
			adtBuilder.Append( CARRIAGE_RETURN );
			adtBuilder.Append( PV1_O);
			adtBuilder.Append( CARRIAGE_RETURN );
			//
			message = PatientUpdateHL7Parser.ParseHL7Message( _hl7Interface, adtBuilder.ToString() );
            Assert.IsNotNull(message, "PatientUpdate Message Parsed");
			Assert.AreEqual("AA", HL7Utility.ParseGetAckCode(message.GetMessage()) , "Ack Check" );
			//
			GetPatientData();
			Assert.AreEqual( _fromPatientSsn , origSsn, "Orig SSN" );
			//
			// *************************************************************************************************************************
			// *************************************************************************************************************************
			// *************************************************************************************************************************
			// TEST #6: CHANGE DOB (then change it back)
			//
			adtBuilder.Remove( 0, adtBuilder.Length );
			//
			string origDob = _fromPatientDob;
			//
			toPatientFirstName = _fromPatientFirstName;
			toPatientLastName = _fromPatientLastName;
			toPatientMiddleName = _fromPatientMiddleName;
			// Sex Code is required in the message but our DB may have null values so let's make sure we are safe
			toPatientSexCode = _fromPatientSexCode == null ? "F" : _fromPatientSexCode;
			toPatientDod = _fromPatientDod;
			//
			toPatientDob = Common.HL7DateFormat.ConvertDateTimeToHL7(DateTime.UtcNow.Subtract(new TimeSpan(7777, 77, 77, 77, 77)));
			//
			_toPatientPidSegment = BuildPid( _fromPatientIcn, _fromPatientDfn, toPatientLastName, toPatientFirstName, toPatientMiddleName, toPatientDob, toPatientSexCode, toPatientSsn, toPatientDod );
			//
			adtBuilder.Append( MSH_A08 );
			adtBuilder.Append( CARRIAGE_RETURN );
			adtBuilder.Append( EVN_A08 );
			adtBuilder.Append( CARRIAGE_RETURN );
			adtBuilder.Append( _toPatientPidSegment );
			adtBuilder.Append( CARRIAGE_RETURN );
			adtBuilder.Append( PV1_O);
			adtBuilder.Append( CARRIAGE_RETURN );
			//
			message = PatientUpdateHL7Parser.ParseHL7Message( _hl7Interface, adtBuilder.ToString() );
            Assert.IsNotNull(message, "PatientUpdate Message Parsed");
			Assert.AreEqual("AA", HL7Utility.ParseGetAckCode(message.GetMessage()) , "Ack Check" );
			//
			// Ensure data was persisted correctly
			GetPatientData();
			Assert.AreEqual( _fromPatientDob , toPatientDob, "DOB" );
			//
			// Re-set Dob
			//
			adtBuilder.Remove( 0, adtBuilder.Length );
			GetPatientData();
			//
			_toPatientPidSegment = BuildPid( _fromPatientIcn, _fromPatientDfn, _fromPatientLastName, _fromPatientFirstName, _fromPatientMiddleName, origDob, _fromPatientSexCode, _fromPatientSsn, string.Empty );
			//
			adtBuilder.Append( MSH_A08 );
			adtBuilder.Append( CARRIAGE_RETURN );
			adtBuilder.Append( EVN_A08 );
			adtBuilder.Append( CARRIAGE_RETURN );
			adtBuilder.Append( _toPatientPidSegment );
			adtBuilder.Append( CARRIAGE_RETURN );
			adtBuilder.Append( PV1_O);
			adtBuilder.Append( CARRIAGE_RETURN );
			//
			message = PatientUpdateHL7Parser.ParseHL7Message( _hl7Interface, adtBuilder.ToString() );
            Assert.IsNotNull(message, "PatientUpdate Message Parsed");
			Assert.AreEqual("AA", HL7Utility.ParseGetAckCode(message.GetMessage()) , "Ack Check" );
			//
			GetPatientData();
			Assert.AreEqual( _fromPatientDob , origDob, "Orig DOB" );
			//
			// *************************************************************************************************************************
			// *************************************************************************************************************************
			// *************************************************************************************************************************
			// TEST #7: CHANGE DOD: ADD (then remove)
			//
			adtBuilder.Remove( 0, adtBuilder.Length );
			//
			toPatientFirstName = _fromPatientFirstName;
			toPatientLastName = _fromPatientLastName;
			toPatientMiddleName = _fromPatientMiddleName;
			// Sex Code is required in the message but our DB may have null values so let's make sure we are safe
			toPatientSexCode = _fromPatientSexCode == null ? "F" : _fromPatientSexCode;
			//
			if ( _fromPatientDod != string.Empty )
			{
				toPatientDod = string.Empty;
			}
			else
			{
				toPatientDod = Common.HL7DateFormat.ConvertDateTimeToHL7(DateTime.UtcNow);
			}
			//
			_toPatientPidSegment = BuildPid( _fromPatientIcn, _fromPatientDfn, toPatientLastName, toPatientFirstName, toPatientMiddleName, toPatientDob, toPatientSexCode, toPatientSsn, toPatientDod );
			//
			adtBuilder.Append( MSH_A08 );
			adtBuilder.Append( CARRIAGE_RETURN );
			adtBuilder.Append( EVN_A08 );
			adtBuilder.Append( CARRIAGE_RETURN );
			adtBuilder.Append( _toPatientPidSegment );
			adtBuilder.Append( CARRIAGE_RETURN );
			adtBuilder.Append( PV1_O);
			adtBuilder.Append( CARRIAGE_RETURN );
			//
			message = PatientUpdateHL7Parser.ParseHL7Message( _hl7Interface, adtBuilder.ToString() );
            Assert.IsNotNull(message, "PatientUpdate Message Parsed");
			Assert.AreEqual("AA", HL7Utility.ParseGetAckCode(message.GetMessage()) , "Ack Check" );
			//
			// Ensure data was persisted correctly
			GetPatientData();
			Assert.AreEqual( _fromPatientDod , toPatientDod, "DOD" );
			//
			// Remove Dod
			//
			adtBuilder.Remove( 0, adtBuilder.Length );
			GetPatientData();
			//
			_toPatientPidSegment = BuildPid( _fromPatientIcn, _fromPatientDfn, _fromPatientLastName, _fromPatientFirstName, _fromPatientMiddleName, _fromPatientDob, _fromPatientSexCode, _fromPatientSsn, string.Empty );
			//
			adtBuilder.Append( MSH_A08 );
			adtBuilder.Append( CARRIAGE_RETURN );
			adtBuilder.Append( EVN_A08 );
			adtBuilder.Append( CARRIAGE_RETURN );
			adtBuilder.Append( _toPatientPidSegment );
			adtBuilder.Append( CARRIAGE_RETURN );
			adtBuilder.Append( PV1_O);
			adtBuilder.Append( CARRIAGE_RETURN );
			//
			message = PatientUpdateHL7Parser.ParseHL7Message( _hl7Interface, adtBuilder.ToString() );
            Assert.IsNotNull(message, "PatientUpdate Message Parsed");
			Assert.AreEqual("AA", HL7Utility.ParseGetAckCode(message.GetMessage()) , "Ack Check" );
			//
			GetPatientData();
			Assert.AreEqual( _fromPatientDod , string.Empty, "Orig DOD");
		}

		/// <summary>
		/// ParseHL7Message - PASS (A11)
		/// </summary>
		[Test]
		public void ParseHL7Message_PASS_ADT_A11()
		{
			StringBuilder adtBuilder = new StringBuilder();
			//
			adtBuilder.Append( MSH_A11 );
			adtBuilder.Append( CARRIAGE_RETURN );
			adtBuilder.Append( EVN_A11 );
			adtBuilder.Append( CARRIAGE_RETURN );
			adtBuilder.Append( GetLocationUpdatePidSegment(false) );
			adtBuilder.Append( CARRIAGE_RETURN );
			adtBuilder.Append( PV1_I);
			adtBuilder.Append( CARRIAGE_RETURN );
			//
			HL7ProtocolMessage message = PatientUpdateHL7Parser.ParseHL7Message( _hl7Interface, adtBuilder.ToString() );
            Assert.IsNotNull(message, "PatientUpdate Message Parsed");
			Assert.AreEqual("AA", HL7Utility.ParseGetAckCode(message.GetMessage()) , "Ack Check" );
		}

		/// <summary>
		/// ParseHL7Message - PASS (A12)
		/// </summary>
		[Test]
		public void ParseHL7Message_PASS_ADT_A12()
		{
			StringBuilder adtBuilder = new StringBuilder();
			//
			adtBuilder.Append( MSH_A12 );
			adtBuilder.Append( CARRIAGE_RETURN );
			adtBuilder.Append( EVN_A12 );
			adtBuilder.Append( CARRIAGE_RETURN );
			adtBuilder.Append( GetLocationUpdatePidSegment(true) );
			adtBuilder.Append( CARRIAGE_RETURN );
			adtBuilder.Append( PV1_I);
			adtBuilder.Append( CARRIAGE_RETURN );
			//
			HL7ProtocolMessage message = PatientUpdateHL7Parser.ParseHL7Message( _hl7Interface, adtBuilder.ToString() );
            Assert.IsNotNull(message, "PatientUpdate Message Parsed");
			Assert.AreEqual("AA", HL7Utility.ParseGetAckCode(message.GetMessage()) , "Ack Check" );
		}

		/// <summary>
		/// ParseHL7Message - PASS (A13)
		/// </summary>
		[Test]
		public void ParseHL7Message_PASS_ADT_A13()
		{
			StringBuilder adtBuilder = new StringBuilder();
			//
			adtBuilder.Append( MSH_A13 );
			adtBuilder.Append( CARRIAGE_RETURN );
			adtBuilder.Append( EVN_A13 );
			adtBuilder.Append( CARRIAGE_RETURN );
			adtBuilder.Append( GetLocationUpdatePidSegment(false) );
			adtBuilder.Append( CARRIAGE_RETURN );
			adtBuilder.Append( PV1_O);
			adtBuilder.Append( CARRIAGE_RETURN );
			//
			HL7ProtocolMessage message = PatientUpdateHL7Parser.ParseHL7Message( _hl7Interface, adtBuilder.ToString() );
            Assert.IsNotNull(message, "PatientUpdate Message Parsed");
			Assert.AreEqual("AA", HL7Utility.ParseGetAckCode(message.GetMessage()) , "Ack Check" );
		}

		/// <summary>
		/// ParseHL7Message - FAIL (interface is null)
		/// </summary>
		[Test]
		[ExpectedException( typeof( ArgumentNullException) )]
		public void ParseHL7Message_Fail_Null_Interface()
		{
			StringBuilder adtBuilder = new StringBuilder();
			//
			adtBuilder.Append( MSH_A13 );
			adtBuilder.Append( CARRIAGE_RETURN );
			adtBuilder.Append( EVN_A13 );
			adtBuilder.Append( CARRIAGE_RETURN );
			adtBuilder.Append( GetLocationUpdatePidSegment(true) );
			adtBuilder.Append( CARRIAGE_RETURN );
			adtBuilder.Append( PV1_O);
			adtBuilder.Append( CARRIAGE_RETURN );
			//
			HL7ProtocolMessage message = PatientUpdateHL7Parser.ParseHL7Message( null, adtBuilder.ToString() );
            Assert.IsNotNull(message, "PatientUpdate Message Parsed");
		}

		/// <summary>
		/// ParseHL7Message - FAIL (message is null)
		/// </summary>
		[Test]
		[ExpectedException( typeof( ArgumentNullException) )]
		public void ParseHL7Message_Fail_Null_Message()
		{
			HL7ProtocolMessage message = PatientUpdateHL7Parser.ParseHL7Message( _hl7Interface, null );
            Assert.IsNotNull(message, "PatientUpdate Message Parsed");
		}

		/// <summary>
		/// ParseHL7Message - FAIL (interface and message are null)
		/// </summary>
		[Test]
		[ExpectedException( typeof( ArgumentNullException) )]
		public void ParseHL7Message_Fail_Null_Interface_And_Message()
		{
			HL7ProtocolMessage message = PatientUpdateHL7Parser.ParseHL7Message( null, null );
            Assert.IsNotNull(message, "PatientUpdate Message Parsed");
		}

		#region Helper Methods

		/// <summary>
		/// 
		/// </summary>
		private string BuildPid( 
			string patientIcn, 
			string patientDfn,
			string patientLastName, 
			string patientFirstName, 
			string patientMiddleName, 
			string patientDob, 
			string patientSexCode, 
			string patientSsn,
			string patientDod )
		{
			StringBuilder pid = new StringBuilder();
			//
			pid.Append( "PID^1^" );
			pid.Append( patientIcn );
			pid.Append( "^" );
			pid.Append( patientDfn );
			pid.Append( "^" );
			pid.Append( "^" );
			pid.Append( patientLastName );
			pid.Append( "~" );
			pid.Append( patientFirstName );
			pid.Append( "~" );
			pid.Append( patientMiddleName );
			pid.Append( "^" );
			pid.Append( "^" );
			pid.Append( patientDob );
			pid.Append( "^" );
			pid.Append( patientSexCode );
			pid.Append( "^" );
			pid.Append( "^" );
			pid.Append( "^" );
			pid.Append( "^" );
			pid.Append( "^" );
			pid.Append( "^" );
			pid.Append( "^" );
			pid.Append( "^" );
			pid.Append( "^" );
			pid.Append( "^" );
			pid.Append( "^" );
			pid.Append( patientSsn );
			//
			if( patientDod != "N/A" )
			{
				pid.Append( "^" );
				pid.Append( "^" );
				pid.Append( "^" );
				pid.Append( "^" );
				pid.Append( "^" );
				pid.Append( "^" );
				pid.Append( "^" );
				pid.Append( "^" );
				pid.Append( "^" );
				pid.Append( "^" );
				pid.Append( patientDod );
			}
			//
			return pid.ToString();
		}

		/// <summary>
		/// 
		/// </summary>
		private void GetPatientData()
		{
			DataRow drPatientData = null;
		    string query = "";
			if ( _patientGuid != Guid.Empty )
			{
				drPatientData = UnitTestUtil.RunSQLGetDataRow( string.Concat("SELECT TOP 1 * FROM Patient WHERE PatientGuid = '", _patientGuid, "' AND PatientSexCode IS NOT NULL") );
			}
			else
			{
				 query = "SELECT TOP 1 * FROM Patient WHERE VistaPatientId IS NOT NULL " +
					"AND PatientSsn IS NOT NULL " +
					"AND PatientFirstName IS NOT NULL " +
					"AND PatientLastName IS NOT NULL " +
					"AND PatientIcn IS NOT NULL ";
				//
				drPatientData = UnitTestUtil.RunSQLGetDataRow( query );
				//
				_patientGuid = (Guid)drPatientData[TABLE.Patient.PatientGuid]; 
			}
			//
            Assert.IsNotNull(drPatientData, query);
			//
			_fromPatientDfn = Convert.ToString(drPatientData[TABLE.Patient.VistaPatientId]).Trim();
			_fromPatientDob = Common.HL7DateFormat.ConvertDateTimeToHL7((DateTime)drPatientData[TABLE.Patient.PatientDob]);
			_fromPatientDod = drPatientData.IsNull(TABLE.Patient.PatientDeathDate) ? string.Empty : Common.HL7DateFormat.ConvertDateTimeToHL7((DateTime)drPatientData[TABLE.Patient.PatientDeathDate]);
			_fromPatientFirstName = Convert.ToString(drPatientData[TABLE.Patient.PatientFirstName]).Trim();
			_fromPatientIcn = Convert.ToString(drPatientData[TABLE.Patient.PatientIcn]).Trim();
			_fromPatientLastName = Convert.ToString(drPatientData[TABLE.Patient.PatientLastName]).Trim();
			_fromPatientMiddleName = Convert.ToString(drPatientData[TABLE.Patient.PatientMiddleName]).Trim();
			_fromPatientSexCode = Convert.ToString(drPatientData[TABLE.Patient.PatientSexCode]).Trim();
			_fromPatientSsn = Convert.ToString(drPatientData[TABLE.Patient.PatientSsn]).Trim();
		}

		/// <summary>
		/// 
		/// </summary>
		private string GetLocationUpdatePidSegment(bool includeIcn)
		{
			string query = "SELECT TOP 1 * FROM Patient WHERE VistaPatientId IS NOT NULL " +
				"AND PatientSsn IS NOT NULL " +
				"AND PatientFirstName IS NOT NULL " +
				"AND PatientLastName IS NOT NULL " +
                "AND PatientSexCode IS NOT NULL ";
			//
			if (includeIcn)
			{
				query += "AND PatientIcn IS NOT NULL ";
			}
			else
			{
				query += "AND PatientIcn IS NULL ";
			}
			//
			DataRow drPatientData = UnitTestUtil.RunSQLGetDataRow( query );
            Assert.IsNotNull(drPatientData, query);
			//
			string patientDfn = Convert.ToString(drPatientData[TABLE.Patient.VistaPatientId]).Trim();
			string patientDob = Common.HL7DateFormat.ConvertDateTimeToHL7((DateTime)drPatientData[TABLE.Patient.PatientDob]);
			string patientFirstName = Convert.ToString(drPatientData[TABLE.Patient.PatientFirstName]).Trim();
			string patientIcn = Convert.ToString(drPatientData[TABLE.Patient.PatientIcn]).Trim();
			string patientLastName = Convert.ToString(drPatientData[TABLE.Patient.PatientLastName]).Trim();
			string patientMiddleName = Convert.ToString(drPatientData[TABLE.Patient.PatientMiddleName]).Trim();
			string patientSexCode = Convert.ToString(drPatientData[TABLE.Patient.PatientSexCode]).Trim();
			string patientSsn = Convert.ToString(drPatientData[TABLE.Patient.PatientSsn]).Trim();
			//
			return BuildPid( patientIcn, patientDfn, patientLastName, patientFirstName, patientMiddleName, patientDob, patientSexCode, patientSsn, "N/A" );
		}

		#endregion
	}
}
#endif